<?php
/**
 * Project: PHP Help Manager
 * @link http://themearabia.net
 * @copyright 2021
 * @author Hossam Hamed <themearabia@gmail.com> <0201094140448>
 * @version 5.0
 */

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Mail;
use App\Models\User;
use hisorange\BrowserDetect\Parser as Browser;
use Illuminate\Support\Str;

class AuthController extends Controller
{

    public function __construct()
    {
        parent::__construct();   
    }

    // singin
    public function login(Request $request)
    {
        if (Auth::check()) {
            return redirect(url('/'));
        }

        if ($request->has('email') && $request->has('password')) {
            $this->validate(
                $request,
                [
                    'email'         => 'required|email',
                    'password'      => 'required',
                ],
                [
                    'email.required'    => lang('error_email_required'),
                    'email.email'       => lang('error_email_correct'),
                    'password.required' => lang('error_password_required'),
                ]
            );

            $credentials = ['email' => $request->get('email'), 'password' => $request->get('password')];
            if (Auth::attempt($credentials, $request->has('remember'))) {
                $browser_detect = parent::browser_detect();
                update_user_meta('browser_detect', maybe_serialize($browser_detect), Auth::user()->id);
                update_user_meta('platformname', Browser::platformName(), Auth::user()->id);
                update_user_meta('browserfamily', Browser::browserFamily(), Auth::user()->id);
                update_user_meta('signintime', time(), Auth::user()->id);
                return redirect(url('/'));
            } else {
                return redirect()->back()->with('errorlogin', 'true');
            }
        }
        else {
            $data['page_title'] = lang('login');
            return get_view('auth.login', $data);
        }
    }

    // forgot password
    public function forgot_password(Request $request)
    {
        $data['page_title'] = lang('forgot_password');
        return get_view('auth.forgot-password', $data);
    }

    // singup
    public function register(Request $request)
    {
        if (Auth::check()) {
            return redirect(url('/'));
        }
        if ($request->has('email') && $request->has('password')) {
            $this->validate(
                $request,
                [
                    'username'              => 'required|unique:users',
                    'email'                 => 'required|email|unique:users',
                    'password'              => 'required|confirmed',
                ],
                [
                    'username.required'     => lang('error_username_required'),
                    'username.unique'       => lang('error_username_unique'),
                    'email.required'        => lang('error_email_required'),
                    'email.unique'          => lang('error_email_unique'),
                    'email.email'           => lang('error_email_correct'),
                    'password.required'     => lang('error_password_required'),
                    'password.confirmed'    => lang('error_password_confirmed'),
                ]
            );

            $active_key = Str::random(32);
            $status     = get_option('user_activate', 0);
            $user = User::create([
                'username'      => safe_input($request->get('username')),
                'email'         => safe_input($request->get('email')),
                'password'      => Hash::make($request->get('password')),
                'pincode'       => '',
                'userlevel'     => 'user',
                'status'        => $status,
                'active_key'    => $active_key,
            ]);
            $browser_detect = parent::browser_detect();
            update_user_meta('signuptime', time(), $user->id);
            update_user_meta('signintime', time(), $user->id);
            update_user_meta('avatar', 'users/default.png', $user->id);
            update_user_meta('browser_detect', maybe_serialize($browser_detect), $user->id);
            update_user_meta('platformname', Browser::platformName(), $user->id);
            update_user_meta('browserfamily', Browser::browserFamily(), $user->id);
            Auth::loginUsingId($user->id);

            $this->username = safe_input($request->get('username'));
            $this->sendto   = safe_input($request->get('email'));

            if($status){

                $mail_data = [
                    'msg_username'  => safe_input($request->get('username')),
                    'msg_sitename'  => get_option('sitename'),
                ];
                
                if(env('IS_MAILER')){
                    Mail::send('emails.welcome', $mail_data, function ($message) {
                        $message->to($this->sendto)->subject(lang('welcome_to', ['sitename' => get_option('sitename')]))->from(get_option('webmaster_email'), get_option('sitename'));
                    });
                }

                return redirect(url('/'));
            }
            else {

                $mail_data = [
                    'msg_link'      => url('active/'.$this->sendto.'/'.$active_key),
                    'msg_emailcode' => $active_key,
                    'msg_username'  => safe_input($request->get('username')),
                    'msg_email'     => $request->get('email')
                ];

                if(env('IS_MAILER')){
                    Mail::send('emails.activate', $mail_data, function ($message) {
                        $message->to($this->sendto)->subject(lang('verify_your_email'))->from(get_option('webmaster_email'), get_option('sitename'));
                    });
                }
                return redirect(url('verification'));
            }
            
        }
        else {
            $data['page_title'] = lang('register');
            return get_view('auth.register', $data);
        }
    }
    
    // index_active_account
    public function index_active_account($email, $code, Request $request){
        $user = DB::table(USERS_TABLE)->where(['email' => safe_input($email), 'active_key' => $code])->get();
        if($user->count()){
            DB::table(USERS_TABLE)->where(['email' => safe_input($email), 'active_key' => $code])->update([
                'active_key' => '',
                'status'    => '1'
            ]);
            $data_user = $user->first();
            $this->username = $data_user->username;
            $this->sendto   = $data_user->email;
            $mail_data = [
                'msg_username'  => safe_input($request->get('username')),
                'msg_sitename'  => get_option('sitename'),
            ];
            if(get_option('email_welcome')){
                Mail::send('emails.welcome', $mail_data, function ($message) {
                    $message->to($this->sendto)->subject(lang('mail_title_welcome', ['sitename' => get_option('sitename')]));
                });
            }
            
            return redirect(url('/'))->with(['activation_success' => lang('account_is_now_activated')]);
        }
        else {
            return redirect(url('/'));
        }
    }

    // logout
    public function logout(Request $request)
    {
        Auth::guard('web')->logout();
        $request->session()->invalidate();
        $request->session()->regenerateToken();
        return redirect('/admincp/login');
    }

    // admin login
    public function adminlogin(Request $request)
    {
        if (Auth::check() && Auth::user()->userlevel == 'admin') {
            return redirect(get_admin_url());
        }
        if ($request->has('email') && $request->has('password') && $request->has('pincode')) {
            $this->validate(
                $request,
                [
                    'email'         => 'required|email',
                    'password'      => 'required',
                    'pincode'       => 'required',
                ],
                [
                    'email.required'    => admin_lang('email_required'),
                    'email.email'       => admin_lang('email_correct'),
                    'password.required' => admin_lang('password_required'),
                    'pincode.required'  => admin_lang('pincode_required'),
                ]
            );
            $roles=['admin', 'supervisor', 'new'];
            $credentials = ['email' => $request->get('email'), 'password' => $request->get('password'), 'pincode' => md5($request->get('pincode')), 'userlevel' => $roles];
            if (Auth::attempt($credentials, $request->has('remember'))) {
                $browser_detect = parent::browser_detect();
                update_user_meta('browser_detect', maybe_serialize($browser_detect), Auth::user()->id);
                update_user_meta('platformname', Browser::platformName(), Auth::user()->id);
                update_user_meta('browserfamily', Browser::browserFamily(), Auth::user()->id);
                update_user_meta('signintime', time(), Auth::user()->id);
                $request->session()->put('auth.pincode_confirmed_at', time());
                return redirect(get_admin_url());
            } else {
                return redirect()->back()->with('errorlogin', 'true');
            }
        } else {
            $data['page_title'] = admin_lang('login');
            return get_admin_view('dashboard_login', $data);
        }
    }
    // index admin pincode confirm
    public function admin_pincode_confirm(Request $request)
    {
        $confirmedAt = time() - $request->session()->get('auth.pincode_confirmed_at', 0);
        $pincodeTimeout = 38800;
        $adminredirecturl = $request->session()->get('adminredirecturl');
        if(!$adminredirecturl) {
            $request->session()->put('adminredirecturl', get_admin_url());
        }
        if(Auth::check() && $confirmedAt > $pincodeTimeout) {
            $data['page_title'] = admin_lang('pincode_confirmed');
            return get_admin_view('pincode_confirmed', $data);
        }
        else {
            $browser_detect = parent::browser_detect();
            update_user_meta('browser_detect', maybe_serialize($browser_detect), Auth::user()->id);
            update_user_meta('platformname', Browser::platformName(), Auth::user()->id);
            update_user_meta('browserfamily', Browser::browserFamily(), Auth::user()->id);
            update_user_meta('signintime', time(), Auth::user()->id);
            return redirect(get_admin_url());
        }
    }
    // admin pincode confirm
    public function admin_pincode_confirm_post(Request $request)
    {
        $confirmedAt = time() - $request->session()->get('auth.pincode_confirmed_at', 0);
        $pincodeTimeout = 38800;
        if(Auth::check() && $confirmedAt > $pincodeTimeout) {
            $this->validate($request, ['pincode' => 'required'], ['pincode.required' => admin_lang('pincode_required')]);
            $user       = Auth::user();
            $pincode    = md5($request->get('pincode'));
            $is_confirm = DB::table(USERS_TABLE)->where(['id' => $user->id, 'pincode' => $pincode])->whereIn('userlevel', ['admin', 'supervisor'])->exists();
            if($is_confirm) {
                $request->session()->put('auth.pincode_confirmed_at', time());
                $adminredirecturl = $request->session()->get('adminredirecturl');
                return ($adminredirecturl)? redirect($adminredirecturl) :   redirect(get_admin_url());
            }
            else {
                return redirect()->back()->with('errorpincode', 'true');
            }
        }
        else {
            $browser_detect = parent::browser_detect();
            update_user_meta('browser_detect', maybe_serialize($browser_detect), Auth::user()->id);
            update_user_meta('platformname', Browser::platformName(), Auth::user()->id);
            update_user_meta('browserfamily', Browser::browserFamily(), Auth::user()->id);
            update_user_meta('signintime', time(), Auth::user()->id);
            return redirect(get_admin_url());
        }
    }
}
