<?php
/**
 * Extension Name: News
 * Extension URI: http://themearabia.net/extension/news
 * Version: 1.0
 * Requires: 5.0
 * Description: News
 * Author: Themearabia
 * Author URI: http://themearabia.net
 */

namespace App\Http\Controllers\Frontend;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;


class NewsController extends Controller
{

    private $type;
    private $url;
    public function __construct()
    {
        parent::__construct();
        $this->type         = 'news';
        $this->url          = url('news');
        $default_options    = [
            "style"                 => "full",
            "column"                => "col-md-4",
            "post_per_page"         => "6",
            "post_per_recent"       => "3",
            "search_text"           => '',
            "widget_search"         => false,
            "widget_tags"           => false,
            "widget_categories"     => false,
            "widget_recent"         => false,
            "meta_shareit"          => false,
        ];
        $options        = maybe_unserialize(site_config('options_news'));
        $this->options  = mega_parse_args($options, $default_options);
    }
    
    // news
    public function news($slug = false, Request $request)
    {
        if(!is_extensions('news')){
            return redirect(url('/'));
        }
        foreach($this->options as $key => $option){
            $data['news_'.$key] = $option;
        }
        $this->paginate = $this->options['post_per_page'];

        if($request->has('search')){
            $search = safe_input($request->get('search'));
        }
        else{
            $search = false;
        }

        $data['the_term_id']    = ($search)? true : false;
        $data['the_cate_name']  = ($search)? lang('search').': '.$search : '';
        $data['post_type']      = $this->type;
        $data['sub_header']     = extension_lang('news', 'news');
        $data['page_title']     = extension_lang('news', 'news');

        if($search){
            $data['breadcrumbs'][]  = ['link' => url('news'), 'title' => extension_lang('news', 'news')];
            $data['breadcrumbs'][]  = ['link' => false, 'title' => lang('search').': '.$search];
        }
        else {
            $data['breadcrumbs'][]  = ['link' => false, 'title' => extension_lang('news', 'news')];
        }
        


        $posts = $this->get_query();

        if($search){
            $posts = $posts->where('post_title', 'like', "%$search%");
        }

        $posts = $posts->orderBy('post_pin', 'desc')
        ->orderBy('post_orders', 'asc')
        ->orderBy('post_modified', 'desc')
        ->paginate($this->paginate);

        if($search){
            $posts->appends(['search' => $search]);
        }

        if ($request->has('page') and $request->get('page') > $posts->lastPage()) {return redirect($posts->url($posts->lastPage()));}
        $data['posts'] = $posts;
        $data = mega_parse_args($data, $this->get_widgets());
        return get_view('news.news', $data);
    }

    // category
    public function category($slug, Request $request)
    {
        if(!is_extensions('news')){
            return redirect(url('/'));
        }
        $slug = utf8_uri_encode(safe_input($slug));
        $term = get_term_by_slug($slug, $this->type);
        if ($term->count()) {
            foreach($this->options as $key => $option){
                $data['news_'.$key] = $option;
            }
            $this->paginate = $this->options['post_per_page'];
            $get_term = $term->first();
            $data['the_term_id']    = $get_term->id;
            $data['the_cate_name']  = $get_term->name;
            $data['post_type']      = $this->type;
            $data['sub_header']     = extension_lang('news', 'news');
            $data['page_title']     = extension_lang('news', 'news').' - '.$get_term->name;
            $data['breadcrumbs'][]  = ['link' => url('news'), 'title' => extension_lang('news', 'news')];
            $data['breadcrumbs'][]  = ['link' => false, 'title' => $get_term->name];
            $posts = $this->get_query()->where(['term_id' => $get_term->id])
            ->orderBy('post_pin', 'desc')
            ->orderBy('post_orders', 'asc')
            ->orderBy('post_modified', 'desc')
            ->paginate($this->paginate);
            if ($request->has('page') and $request->get('page') > $posts->lastPage()) {return redirect($posts->url($posts->lastPage()));}
            $data['posts'] = $posts;
            $data = mega_parse_args($data, $this->get_widgets());
            return get_view('news.news', $data);
        }
        else {
            return redirect($this->url);
        }
    }

    // tags
    public function tags($slug, Request $request)
    {
        if(!is_extensions('news')){
            return redirect(url('/'));
        }
        $slug = safe_input($slug);
        foreach($this->options as $key => $option){
            $data['news_'.$key] = $option;
        }
        $this->paginate = $this->options['post_per_page'];
        $data['the_term_id']    = false;
        $data['the_cate_name']  = lang('tag').': '.$slug;
        $data['post_type']      = $this->type;
        $data['sub_header']     = extension_lang('news', 'news');
        $data['page_title']     = extension_lang('news', 'news').' - '.$slug;
        $data['breadcrumbs'][]  = ['link' => url('news'), 'title' => extension_lang('news', 'news')];
        $data['breadcrumbs'][]  = ['link' => false, 'title' => lang('tag').': '.$slug];
        $posts = $this->get_query()->where('post_tags', 'like', "%$slug%")
        ->orderBy('post_pin', 'desc')
        ->orderBy('post_orders', 'asc')
        ->orderBy('post_modified', 'desc')
        ->paginate($this->paginate);
        if ($request->has('page') and $request->get('page') > $posts->lastPage()) {return redirect($posts->url($posts->lastPage()));}
        $data['posts'] = $posts;
        $data = mega_parse_args($data, $this->get_widgets());
        return get_view('news.news', $data);
    }

    // single
    public function single($slug, Request $request)
    {
        if(!is_extensions('news')){
            return redirect(url('/'));
        }
        $slug = utf8_uri_encode(safe_input($slug));
        $post = DB::table(POSTS_TABLE)->where(['post_status' => '1', 'post_type' => $this->type, 'post_name' => $slug])->get();
        if ($post->count()) {
            foreach($this->options as $key => $option){
                $data['news_'.$key] = $option;
            }
            $single     = $post->first();
            $views      = $single->post_views+1;
            DB::table(POSTS_TABLE)->where('id', $single->id)->update(['post_views' => $views]);
            $data['the_term_id']    = $single->term_id;
            $data['single']         = $single;
            $data['page_title']     = $single->post_title;
            $data['sub_header']     = $single->post_title;
            $data['post_type']      = $this->type;
            $data['breadcrumbs'][]  = ['link' => url('news'), 'title' => extension_lang('news', 'news')];
            $data['breadcrumbs'][]  = ['link' => false, 'title' => $single->post_title];
            $data['post_meta']      = query_posts_meta($single->id);
            $data['tags']           = get_tags_args($single->post_tags, 'news/tag/');
            $comments               = get_loop_comments_single($single->post_type, $single->id, $request);
            $data['comments']       = $comments;
            $data['description']    = $data['post_meta']['meta_desc'];
            $data['keywords']       = $single->post_tags;
            $data = mega_parse_args($data, $this->get_widgets());
            return get_view('news.single_new', $data);
        }
        else {
            return redirect($this->url);
        }
    }

    // widgets
    public function get_widgets($single_id = false)
    {
        $data['widget_categories'] = DB::table('terms')
        ->leftJoin('posts', 'posts'.'.term_id', '=', 'terms'.'.id')
        ->selectRaw(
            db_select_column_as('terms', 'name').
            db_select_column_as('terms', 'slug').
            db_select_column_as('terms', 'id').
            db_select_column_as_count('posts', 'id', 'posts_count', '')
        )
        ->where(['type' => $this->type, 'status' => '1', 'post_status' => '1', 'post_type' => $this->type])
        ->groupBy('posts'.'.term_id')->orderBy('orders', 'asc')->get();


        $data['widget_recent'] = $this->get_query()
        ->orderBy('post_modified', 'desc')
        ->limit($this->options['post_per_recent'])
        ->get();

        $data['widget_tags'] = get_post_type_tags($this->type);

        return $data;
    }
    
    // query
    public function get_query()
    {
        $query = DB::table('posts')
        ->leftJoin('postsmeta as thumbnails', function ($join) {$join->on('thumbnails.post_id', '=', 'posts.id')->where('thumbnails.meta_key', '=', 'thumbnails');})
        ->leftJoin('users', function ($join) {$join->on('users.id', '=', 'posts.post_author');})
        ->leftJoin('terms', function ($join) {$join->on('terms.id', '=', 'posts.term_id');})
        ->selectRaw(
            db_select_column_as('posts', 'post_title').
            db_select_column_as('posts', 'post_name').
            db_select_column_as('posts', 'post_excerpts').
            db_select_column_as('posts', 'id', 'post_id').
            db_select_column_as('posts', 'post_author').
            db_select_column_as('posts', 'post_type').
            db_select_column_as('posts', 'term_id').
            db_select_column_as('posts', 'comment_status').
            db_select_column_as('posts', 'post_modified').
            db_select_column_as('posts', 'post_views').
            db_select_column_as('terms', 'slug').
            db_select_column_as('terms', 'name').
            db_select_column_as('users', 'username').
            db_select_column_as('users', 'id', 'author').
            db_select_column_as('thumbnails', 'meta_value', 'thumbnail', '')
        )
        ->where(['post_status' => '1', 'post_type' => $this->type]);
        return $query;
    }
    
}
